/*
 * Copyright (c) Doug Palmer <doug@charvolant.org> 2005
 *
 * See LICENSE for licensing details.
 * 
 * $Id$
 */

package org.charvolant.sudoku;

import java.util.ArrayList;
import java.util.Collection;

/**
 * A shape on the sudoku board that must have a
 * set of unique cells.
 * <p>
 * This is an abstract class that covers the basic
 * shape constraints. Subclasses are responsible for
 * specifc shapes.
 * 
 * @author doug
 *
 */
public abstract class Shape extends Model {
  /** The enclosing board */
  private Board board;
  /** The cells within the shape */
  private Cell[] cells;
  /** Do a singleton search? */
  private boolean search;
  
  public Shape(Board board) {
    this.board = board;
    this.reset();
  }
  
  /**
   * Get the size of a side of thge square.
   * <p>
   * This is the same as the board size.
   * 
   * @return The side size.
   */
  public int getSize() {
    return this.board.getSize();
  }
  
  /* 
   * Get the context that this square uses.
   * 
   * @see org.charvolant.sudoku.Board#getContext()
   * 
   * @return The context
   */
  public Context getContext() {
    return this.board.getContext();
  }
  
  /**
   * Reset the cells in this shape.
   */
  public void reset() {
    int size = this.getSize();
    
    this.cells = new Cell[size];
    this.search = false;
  }
  
  /**
   * Set the search flag.
   * <p>
   * This will allow a search to be make on this shape later.
   */
  public void setSearch() {
    this.search = true;
  }

  /**
   * Add a cell to the list of cells.
   * 
   * @param cell The cell
   */
  public void addCell(Cell cell) {
    for (int i = 0; i < this.cells.length; i++) {
      if (this.cells[i] == null) {
        this.cells[i] = cell;
        return;
      }
    }
    throw new IllegalStateException("Run out of places to put " + cell);
  }
  
  /**
   * A cell has been fixed.
   * <p>
   * Eliminate that possibility from the other cells and
   * then check to see if we have a singleton.
   * 
   * @param cell The cell that has been fixed.
   */
  public void fixed(Cell cell) {
    int size = this.getSize();
    int value = cell.getValue();
    int i;
    
    assert cell.isFixed();
    for (i = 0; i < size; i++)
      this.cells[i].eliminate(value);
    this.setSearch();
  }
  
  /**
   * See if this shape has any value reduced to a singleton.
   * <p>
   * If it has, then the appropriate cell is marked.
   * <p>
   * This search is only undertaken if {@link #setSearch()} was called.
   * Performing the search resets the search flag.
   */
  public void search() {
    int size = this.getSize();
    int i, v, count;
    Cell cell, candidate;
    boolean fixed;
    
    if (!this.search)
      return;
    for (v = 0; v < size; v++) {
      count = 0;
      candidate = null;
      fixed = false;
      for (i = 0; i < size && !fixed; i++) {
        cell = this.cells[i];
        if (cell.isFixed()) {
          if (cell.getValue() == v)
            fixed = true;
        } else {
          if (cell.isPossible(v)) {
            count++;
            candidate = cell;
          }
        }
      }
      if (!fixed && count == 1 && candidate != null)
        candidate.singleton(v);
    }
    this.search = false;
  }
  
  /**
   * Get the singleton cells.
   * 
   * @return A collection of singleton cells
   */
  public Collection<Cell> getSingletons() {
    Collection<Cell> singletons = new ArrayList<Cell>();
    int size = this.getSize();
    
    for (int i = 0; i < size; i++)
      if (this.cells[i].isSingleton())
        singletons.add(this.cells[i]);
    return singletons;
  }
  
  /**
   * Get the cells.
   * 
   * @return A collection of the squares cells
   */
  public Collection<Cell> getCells() {
    Collection<Cell> cells = new ArrayList<Cell>();
    int size = this.getSize();
    
    for (int i = 0; i < size; i++)
      cells.add(this.cells[i]);
    return cells;
  }
  
}
