/*
 * Copyright (c) Doug Palmer <doug@charvolant.org> 2005
 *
 * See LICENSE for licensing details.
 * 
 * $Id$
 */

package org.charvolant.sudoku.gui;

import java.awt.*;
import java.awt.event.*;

import org.charvolant.sudoku.*;

/**
 * A canvas that shows the state of a cell.
 * 
 * @author doug
 */
public class CellCanvas extends Canvas implements MouseListener, ModelListener {
  /** The cell */
  private Cell cell;
  /** The border colour */
  private DisplayPreferences preferences;
  /** The popup menu */
  private PopupMenu popup;
  /** The popup menu items */
  private MenuItem[] popupValues;
  
  /**
   * Construct for an underlying cell.
   * 
   * @param cell The cell
   * @param preferences The display preferences
   */
  public CellCanvas(Cell cell, DisplayPreferences preferences) {
    this.cell = cell;
    this.cell.addListener(this);
    this.preferences = preferences;
    this.setBackground(this.preferences.getBackground());
    this.addMouseListener(this);
    this.popup = new PopupMenu();
    boolean[] possibles = this.cell.getPossibilities();
    this.popupValues = new MenuItem[possibles.length];
    for (int v = 0; v < possibles.length; v++) {
      this.popupValues[v] = new MenuItem(this.preferences.getValueName(v));
      this.popupValues[v].addActionListener(new FixValue(v));
      popup.add(this.popupValues[v]);
    }
    this.add(this.popup);
  }
  
  /** 
   * Get the minimum size for the canvas.
   * <p>
   * This is enough to display a 2x2 pixel space for each possibility, 
   * along with a borde.
   */
  @Override
  public Dimension getMinimumSize() {
    int height = this.cell.getSquareSize() * 3 + 3;
    
    return new Dimension(height, height);
  }
  
  /** 
   * Get the preferred size for the canvas.
   * <p>
   * This is enough to display a 3x3 pixel space for each possibility, 
   * along with a borde.
   */
  @Override
  public Dimension getPreferredSize() {
    int height = this.cell.getSquareSize() * 10 + 3;
    
    return new Dimension(height, height);
  }
  
  /**
   * Paint this cell.
   * 
   * @param g The graphics context
   */
  public void paint(Graphics g) {
    Dimension size = this.getSize();
    int cellSize = this.cell.getSquareSize();
    
    if (g instanceof Graphics2D) {
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
    }
    if (this.cell.isFixed()) {
      Font font = this.preferences.getFont();
      String value = this.preferences.getValueName(this.cell.getValue());
      FontMetrics fm = g.getFontMetrics(font);
      int height = fm.getAscent() + fm.getDescent();
      int width = fm.stringWidth(value);
      if (height < size.height * 2 / 3 && width < size.width * 2 / 3) {
        int ptx = font.getSize() * size.width * 2 / (3 * width);
        int pty = font.getSize() * size.height * 2 / (3 * height);
        font = new Font(font.getFontName(), font.getStyle(), Math.min(ptx, pty));
        fm = g.getFontMetrics(font);
        height = fm.getAscent() + fm.getDescent();
        width = fm.stringWidth(value);
      }
      g.setFont(font);
      g.setColor(this.preferences.getFixed());
      g.drawString(value, (size.width - width) / 2, (size.height - height) / 2 + fm.getAscent());
    } else if (this.cell.isEmpty()) {
      g.setColor(this.preferences.getEmpty());
      g.fillRect(0, 0, size.width, size.height);
    } else {
      int v = 0;
      for (int i = 0; i < cellSize; i++)
        for (int j = 0; j < cellSize; j++) {
          if (this.cell.getSingleton() == v)
            g.setColor(this.preferences.getSingleton());
          else
            g.setColor(this.cell.isPossible(v) ? this.preferences.getPossible() : this.preferences.getEliminated());
          int x = j * (size.width - cellSize - 2) / cellSize + 3;
          int y = i * (size.height - cellSize - 2) / cellSize + 3;  
          int x1 = (j + 1) * (size.width - cellSize - 2) / cellSize + 3;
          int y1 = (i + 1) * (size.height - cellSize - 2) / cellSize + 3;  
          g.fillRect(x, y, x1 - x - 1, y1 - y - 1);
          v++;
        }
    }
    g.setColor(this.preferences.getBorder());
    g.drawLine(0, 0, size.width - 1, 0);
    g.drawLine(size.width - 1, 0, size.width - 1, size.height - 1);
    g.drawLine(size.width - 1, size.height - 1, 0, size.height - 1);
    g.drawLine(0, size.height - 1, 0, 0);
  }
  
  /**
   * The mouse has been clicked.
   * <p>
   * Ignore this.
   *
   * @param e The mouse event
   *
   * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
   */
  public void mouseClicked(MouseEvent e) {
  }
  
  /**
   * The mouse has entered the canvas.
   * <p>
   * Ignore this
   *
   * @param e The event
   *
   * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
   */
  public void mouseEntered(MouseEvent e) {
  }
  
  /**
   * The mouse has exited the canvas.
   * <p>
   * Ignore this
   *
   * @param e The event
   *
   * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
   */
  public void mouseExited(MouseEvent e) {
  }
  
  /**
   * The mouse button has been pressed.
   * <p>
   * If this is the "right" button then display a pop-up.
   *
   * @param e The mouse event
   *
   * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
   */
  public void mousePressed(MouseEvent e) {
    if (e.isPopupTrigger() && !this.cell.isFixed())
      this.showMenu(e);
  }
  
  /**
   * The mouse button has been released.
   * <p>
   * If this is the "right" button then display a pop-up.
   *
   * @param e The event
   *
   * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
   */
  public void mouseReleased(MouseEvent e) {
    if (e.isPopupTrigger() && !this.cell.isFixed())
      this.showMenu(e);    
  }
  
  private void showMenu(MouseEvent e) {
    boolean[] possibles = this.cell.getPossibilities();
    
    for (int v = 0; v < possibles.length; v++)
      this.popupValues[v].setEnabled(possibles[v]);
    popup.show(e.getComponent(), e.getX(), e.getY());
  }
  
  /**
   * Receive a model update.
   * <p>
   * Repaint the cell.
   *
   * @param event The update event
   *
   * @see org.charvolant.sudoku.ModelListener#modelUpdate(org.charvolant.sudoku.ModelEvent)
   */
  public void modelUpdate(ModelEvent event) {
    this.repaint();
  }
 
  
  private class FixValue implements ActionListener {
    /** The value to fix to */
    private int value;
    
    public FixValue(int value) {
      this.value = value;
    }
    
    /**
     * Set the value.
     *
     * @param e The event that caused this
     *
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent e) {
      CellCanvas.this.cell.getContext().choiceUser();
      CellCanvas.this.cell.fix(this.value);
      CellCanvas.this.cell.fireSearch();
    }
  }
}
